/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.io.sstable.metadata;

import com.google.common.base.Throwables;
import com.google.common.collect.Lists;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.Collections;
import java.util.EnumMap;
import java.util.EnumSet;
import java.util.Map;
import java.util.function.UnaryOperator;
import java.util.zip.CRC32;
import org.apache.cassandra.io.FSWriteError;
import org.apache.cassandra.io.sstable.CorruptSSTableException;
import org.apache.cassandra.io.sstable.Descriptor;
import org.apache.cassandra.io.sstable.format.SSTableFormat;
import org.apache.cassandra.io.sstable.format.Version;
import org.apache.cassandra.io.sstable.metadata.IMetadataSerializer;
import org.apache.cassandra.io.sstable.metadata.MetadataCollector;
import org.apache.cassandra.io.sstable.metadata.MetadataComponent;
import org.apache.cassandra.io.sstable.metadata.MetadataType;
import org.apache.cassandra.io.sstable.metadata.StatsMetadata;
import org.apache.cassandra.io.util.DataInputBuffer;
import org.apache.cassandra.io.util.DataOutputBuffer;
import org.apache.cassandra.io.util.DataOutputPlus;
import org.apache.cassandra.io.util.File;
import org.apache.cassandra.io.util.FileDataInput;
import org.apache.cassandra.io.util.FileOutputStreamPlus;
import org.apache.cassandra.io.util.RandomAccessReader;
import org.apache.cassandra.utils.FBUtilities;
import org.apache.cassandra.utils.TimeUUID;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MetadataSerializer
implements IMetadataSerializer {
    private static final Logger logger = LoggerFactory.getLogger(MetadataSerializer.class);
    private static final int CHECKSUM_LENGTH = 4;

    @Override
    public void serialize(Map<MetadataType, MetadataComponent> components, DataOutputPlus out, Version version) throws IOException {
        boolean checksum = version.hasMetadataChecksum();
        CRC32 crc = new CRC32();
        ArrayList sortedComponents = Lists.newArrayList(components.values());
        Collections.sort(sortedComponents);
        out.writeInt(components.size());
        FBUtilities.updateChecksumInt(crc, components.size());
        MetadataSerializer.maybeWriteChecksum(crc, out, version);
        int lastPosition = 4 + 8 * sortedComponents.size() + (checksum ? 8 : 0);
        EnumMap<MetadataType, Integer> sizes = new EnumMap<MetadataType, Integer>(MetadataType.class);
        for (MetadataComponent component : sortedComponents) {
            MetadataType type = component.getType();
            out.writeInt(type.ordinal());
            FBUtilities.updateChecksumInt(crc, type.ordinal());
            out.writeInt(lastPosition);
            FBUtilities.updateChecksumInt(crc, lastPosition);
            int size = type.serializer.serializedSize(version, component);
            lastPosition += size + (checksum ? 4 : 0);
            sizes.put(type, size);
        }
        MetadataSerializer.maybeWriteChecksum(crc, out, version);
        for (MetadataComponent component : sortedComponents) {
            byte[] bytes;
            try (DataOutputBuffer dob = new DataOutputBuffer((Integer)sizes.get((Object)component.getType()));){
                component.getType().serializer.serialize(version, component, dob);
                bytes = dob.getData();
            }
            out.write(bytes);
            crc.reset();
            crc.update(bytes);
            MetadataSerializer.maybeWriteChecksum(crc, out, version);
        }
    }

    private static void maybeWriteChecksum(CRC32 crc, DataOutputPlus out, Version version) throws IOException {
        if (version.hasMetadataChecksum()) {
            out.writeInt((int)crc.getValue());
        }
    }

    @Override
    public Map<MetadataType, MetadataComponent> deserialize(Descriptor descriptor, EnumSet<MetadataType> types) throws IOException {
        EnumMap<MetadataType, MetadataComponent> components;
        logger.trace("Load metadata for {}", (Object)descriptor);
        File statsFile = descriptor.fileFor(SSTableFormat.Components.STATS);
        if (!statsFile.exists()) {
            logger.trace("No sstable stats for {}", (Object)descriptor);
            components = new EnumMap<MetadataType, StatsMetadata>(MetadataType.class);
            components.put(MetadataType.STATS, MetadataCollector.defaultStatsMetadata());
        } else {
            try (RandomAccessReader r = RandomAccessReader.open(statsFile);){
                components = this.deserialize(descriptor, r, types);
            }
        }
        return components;
    }

    @Override
    public MetadataComponent deserialize(Descriptor descriptor, MetadataType type) throws IOException {
        return this.deserialize(descriptor, EnumSet.of(type)).get((Object)type);
    }

    public Map<MetadataType, MetadataComponent> deserialize(Descriptor descriptor, FileDataInput in, EnumSet<MetadataType> selectedTypes) throws IOException {
        int i;
        boolean isChecksummed = descriptor.version.hasMetadataChecksum();
        CRC32 crc = new CRC32();
        int length = (int)in.bytesRemaining();
        int count = in.readInt();
        FBUtilities.updateChecksumInt(crc, count);
        MetadataSerializer.maybeValidateChecksum(crc, in, descriptor);
        int[] ordinals = new int[count];
        int[] offsets = new int[count];
        int[] lengths = new int[count];
        for (i = 0; i < count; ++i) {
            ordinals[i] = in.readInt();
            FBUtilities.updateChecksumInt(crc, ordinals[i]);
            offsets[i] = in.readInt();
            FBUtilities.updateChecksumInt(crc, offsets[i]);
        }
        MetadataSerializer.maybeValidateChecksum(crc, in, descriptor);
        lengths[count - 1] = length - offsets[count - 1];
        for (i = 0; i < count - 1; ++i) {
            lengths[i] = offsets[i + 1] - offsets[i];
        }
        MetadataType[] allMetadataTypes = MetadataType.values();
        EnumMap<MetadataType, MetadataComponent> components = new EnumMap<MetadataType, MetadataComponent>(MetadataType.class);
        for (int i2 = 0; i2 < count; ++i2) {
            MetadataType type = allMetadataTypes[ordinals[i2]];
            if (!selectedTypes.contains((Object)type)) {
                in.skipBytes(lengths[i2]);
                continue;
            }
            byte[] buffer = new byte[isChecksummed ? lengths[i2] - 4 : lengths[i2]];
            in.readFully(buffer);
            crc.reset();
            crc.update(buffer);
            MetadataSerializer.maybeValidateChecksum(crc, in, descriptor);
            try (DataInputBuffer dataInputBuffer = new DataInputBuffer(buffer);){
                components.put(type, type.serializer.deserialize(descriptor.version, dataInputBuffer));
                continue;
            }
        }
        return components;
    }

    private static void maybeValidateChecksum(CRC32 crc, FileDataInput in, Descriptor descriptor) throws IOException {
        int expectedChecksum;
        if (!descriptor.version.hasMetadataChecksum()) {
            return;
        }
        int actualChecksum = (int)crc.getValue();
        if (actualChecksum != (expectedChecksum = in.readInt())) {
            File file = descriptor.fileFor(SSTableFormat.Components.STATS);
            throw new CorruptSSTableException((Throwable)new IOException("Checksums do not match for " + file), file);
        }
    }

    @Override
    public void mutate(Descriptor descriptor, String description, UnaryOperator<StatsMetadata> transform) throws IOException {
        if (logger.isTraceEnabled()) {
            logger.trace("Mutating {} to {}", (Object)descriptor.fileFor(SSTableFormat.Components.STATS), (Object)description);
        }
        this.mutate(descriptor, transform);
    }

    @Override
    public void mutateLevel(Descriptor descriptor, int newLevel) throws IOException {
        if (logger.isTraceEnabled()) {
            logger.trace("Mutating {} to level {}", (Object)descriptor.fileFor(SSTableFormat.Components.STATS), (Object)newLevel);
        }
        this.mutate(descriptor, stats -> stats.mutateLevel(newLevel));
    }

    @Override
    public void mutateRepairMetadata(Descriptor descriptor, long newRepairedAt, TimeUUID newPendingRepair, boolean isTransient) throws IOException {
        if (logger.isTraceEnabled()) {
            logger.trace("Mutating {} to repairedAt time {} and pendingRepair {}", new Object[]{descriptor.fileFor(SSTableFormat.Components.STATS), newRepairedAt, newPendingRepair});
        }
        this.mutate(descriptor, stats -> stats.mutateRepairedMetadata(newRepairedAt, newPendingRepair, isTransient));
    }

    private void mutate(Descriptor descriptor, UnaryOperator<StatsMetadata> transform) throws IOException {
        Map<MetadataType, MetadataComponent> currentComponents = this.deserialize(descriptor, EnumSet.allOf(MetadataType.class));
        StatsMetadata stats = (StatsMetadata)currentComponents.remove((Object)MetadataType.STATS);
        currentComponents.put(MetadataType.STATS, (MetadataComponent)transform.apply(stats));
        this.rewriteSSTableMetadata(descriptor, currentComponents);
    }

    @Override
    public void rewriteSSTableMetadata(Descriptor descriptor, Map<MetadataType, MetadataComponent> currentComponents) throws IOException {
        File file = descriptor.tmpFileFor(SSTableFormat.Components.STATS);
        try (FileOutputStreamPlus out = file.newOutputStream(File.WriteMode.OVERWRITE);){
            this.serialize(currentComponents, out, descriptor.version);
            ((OutputStream)out).flush();
        }
        catch (IOException e) {
            Throwables.throwIfInstanceOf((Throwable)e, FileNotFoundException.class);
            throw new FSWriteError((Throwable)e, file);
        }
        file.move(descriptor.fileFor(SSTableFormat.Components.STATS));
    }
}

