/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.ozone.container.keyvalue;

import com.google.common.base.Preconditions;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.file.NoSuchFileException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Optional;
import org.apache.hadoop.hdds.StringUtils;
import org.apache.hadoop.hdds.conf.ConfigurationSource;
import org.apache.hadoop.hdds.protocol.datanode.proto.ContainerProtos;
import org.apache.hadoop.hdds.scm.container.common.helpers.StorageContainerException;
import org.apache.hadoop.hdfs.util.Canceler;
import org.apache.hadoop.hdfs.util.DataTransferThrottler;
import org.apache.hadoop.ozone.common.Checksum;
import org.apache.hadoop.ozone.common.ChecksumData;
import org.apache.hadoop.ozone.common.OzoneChecksumException;
import org.apache.hadoop.ozone.container.checksum.ContainerMerkleTreeWriter;
import org.apache.hadoop.ozone.container.common.helpers.BlockData;
import org.apache.hadoop.ozone.container.common.helpers.ChunkInfo;
import org.apache.hadoop.ozone.container.common.helpers.ContainerUtils;
import org.apache.hadoop.ozone.container.common.impl.ContainerDataYaml;
import org.apache.hadoop.ozone.container.common.impl.ContainerLayoutVersion;
import org.apache.hadoop.ozone.container.common.interfaces.BlockIterator;
import org.apache.hadoop.ozone.container.common.interfaces.DBHandle;
import org.apache.hadoop.ozone.container.common.volume.HddsVolume;
import org.apache.hadoop.ozone.container.keyvalue.KeyValueContainer;
import org.apache.hadoop.ozone.container.keyvalue.KeyValueContainerData;
import org.apache.hadoop.ozone.container.keyvalue.helpers.BlockUtils;
import org.apache.hadoop.ozone.container.keyvalue.helpers.ChunkUtils;
import org.apache.hadoop.ozone.container.keyvalue.helpers.KeyValueContainerLocationUtil;
import org.apache.hadoop.ozone.container.ozoneimpl.ContainerScanError;
import org.apache.hadoop.ozone.container.ozoneimpl.DataScanResult;
import org.apache.hadoop.ozone.container.ozoneimpl.MetadataScanResult;
import org.apache.hadoop.util.DirectBufferPool;
import org.apache.ratis.thirdparty.com.google.protobuf.ByteString;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class KeyValueContainerCheck {
    private static final Logger LOG = LoggerFactory.getLogger(KeyValueContainerCheck.class);
    private final long containerID;
    private final ConfigurationSource checkConfig;
    private final String metadataPath;
    private final HddsVolume volume;
    private final KeyValueContainer container;
    private final KeyValueContainerData containerDataFromMemory;
    private KeyValueContainerData containerDataFromDisk;
    private static final DirectBufferPool BUFFER_POOL = new DirectBufferPool();

    public KeyValueContainerCheck(ConfigurationSource conf, KeyValueContainer container) {
        this.checkConfig = conf;
        this.container = container;
        this.containerDataFromDisk = null;
        this.containerDataFromMemory = this.container.getContainerData();
        this.containerID = this.containerDataFromMemory.getContainerID();
        this.metadataPath = this.containerDataFromMemory.getMetadataPath();
        this.volume = this.containerDataFromMemory.getVolume();
    }

    public MetadataScanResult fastCheck() throws InterruptedException {
        LOG.debug("Running metadata checks for container {}", (Object)this.containerID);
        try {
            List<ContainerScanError> metadataErrors = this.scanMetadata();
            if (this.containerIsDeleted()) {
                MetadataScanResult metadataScanResult = MetadataScanResult.deleted();
                return metadataScanResult;
            }
            MetadataScanResult metadataScanResult = MetadataScanResult.fromErrors(metadataErrors);
            return metadataScanResult;
        }
        finally {
            if (Thread.currentThread().isInterrupted()) {
                throw new InterruptedException("Metadata scan of container " + this.containerID + " interrupted.");
            }
        }
    }

    private List<ContainerScanError> scanMetadata() {
        ArrayList<ContainerScanError> metadataErrors = new ArrayList<ContainerScanError>();
        File containerDir = new File(this.metadataPath).getParentFile();
        if (!containerDir.exists()) {
            metadataErrors.add(new ContainerScanError(ContainerScanError.FailureType.MISSING_CONTAINER_DIR, containerDir, new FileNotFoundException("Container directory " + containerDir + " not found.")));
            return metadataErrors;
        }
        File metadataDir = new File(this.metadataPath);
        if (!metadataDir.exists()) {
            metadataErrors.add(new ContainerScanError(ContainerScanError.FailureType.MISSING_METADATA_DIR, metadataDir, new FileNotFoundException("Metadata directory " + metadataDir + " not found.")));
            return metadataErrors;
        }
        File containerFile = KeyValueContainer.getContainerFile(this.metadataPath, this.containerID);
        try {
            this.loadContainerData(containerFile);
        }
        catch (FileNotFoundException | NoSuchFileException ex) {
            metadataErrors.add(new ContainerScanError(ContainerScanError.FailureType.MISSING_CONTAINER_FILE, containerFile, ex));
            return metadataErrors;
        }
        catch (IOException ex) {
            metadataErrors.add(new ContainerScanError(ContainerScanError.FailureType.CORRUPT_CONTAINER_FILE, containerFile, ex));
            return metadataErrors;
        }
        metadataErrors.addAll(this.checkContainerFile(containerFile));
        File chunksDir = new File(this.containerDataFromDisk.getChunksPath());
        if (!chunksDir.exists()) {
            metadataErrors.add(new ContainerScanError(ContainerScanError.FailureType.MISSING_CHUNKS_DIR, chunksDir, new FileNotFoundException("Chunks directory " + chunksDir + " not found.")));
        }
        return metadataErrors;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public DataScanResult fullCheck(DataTransferThrottler throttler, Canceler canceler) throws InterruptedException {
        MetadataScanResult metadataResult = this.fastCheck();
        if (metadataResult.isDeleted()) {
            return DataScanResult.deleted();
        }
        if (metadataResult.hasErrors()) {
            return DataScanResult.unhealthyMetadata(metadataResult);
        }
        LOG.debug("Running data checks for container {}", (Object)this.containerID);
        try {
            ContainerMerkleTreeWriter dataTree = new ContainerMerkleTreeWriter();
            List<ContainerScanError> dataErrors = this.scanData(dataTree, throttler, canceler);
            if (this.containerIsDeleted()) {
                DataScanResult dataScanResult = DataScanResult.deleted();
                return dataScanResult;
            }
            DataScanResult dataScanResult = DataScanResult.fromErrors(dataErrors, dataTree);
            return dataScanResult;
        }
        finally {
            if (Thread.currentThread().isInterrupted()) {
                throw new InterruptedException("Data scan of container " + this.containerID + " interrupted.");
            }
        }
    }

    private List<ContainerScanError> scanData(ContainerMerkleTreeWriter currentTree, DataTransferThrottler throttler, Canceler canceler) {
        Preconditions.checkState((this.containerDataFromDisk != null ? 1 : 0) != 0, (Object)"invoke loadContainerData prior to calling this function");
        ArrayList<ContainerScanError> errors = new ArrayList<ContainerScanError>();
        File dbFile = this.containerDataFromDisk.getDbFile();
        if (!dbFile.exists() || !dbFile.canRead()) {
            String dbFileErrorMsg = "Unable to access DB File [" + dbFile.toString() + "] for Container [" + this.containerID + "] metadata path [" + this.metadataPath + "]";
            errors.add(new ContainerScanError(ContainerScanError.FailureType.INACCESSIBLE_DB, dbFile, new IOException(dbFileErrorMsg)));
            return errors;
        }
        try (DBHandle db = BlockUtils.getDB(this.containerDataFromDisk, this.checkConfig);
             BlockIterator<BlockData> kvIter = db.getStore().getBlockIterator(this.containerDataFromDisk.getContainerID(), this.containerDataFromDisk.getUnprefixedKeyFilter());){
            while (kvIter.hasNext() && !this.containerIsDeleted()) {
                List<ContainerScanError> blockErrors = this.scanBlock(db, dbFile, kvIter.nextBlock(), throttler, canceler, currentTree);
                errors.addAll(blockErrors);
            }
        }
        catch (IOException ex) {
            errors.add(new ContainerScanError(ContainerScanError.FailureType.INACCESSIBLE_DB, dbFile, ex));
        }
        return errors;
    }

    private List<ContainerScanError> checkContainerFile(File containerFile) {
        String dbType;
        String errStr;
        Preconditions.checkState((this.containerDataFromDisk != null ? 1 : 0) != 0, (Object)"Container File not loaded");
        ArrayList<ContainerScanError> errors = new ArrayList<ContainerScanError>();
        try {
            ContainerUtils.verifyContainerFileChecksum(this.containerDataFromDisk, this.checkConfig);
        }
        catch (IOException ex) {
            errors.add(new ContainerScanError(ContainerScanError.FailureType.CORRUPT_CONTAINER_FILE, containerFile, ex));
            return errors;
        }
        if (this.containerDataFromDisk.getContainerType() != ContainerProtos.ContainerType.KeyValueContainer) {
            errStr = "Bad Container type in Containerdata for " + this.containerID;
            errors.add(new ContainerScanError(ContainerScanError.FailureType.CORRUPT_CONTAINER_FILE, containerFile, new IOException(errStr)));
        }
        if (this.containerDataFromDisk.getContainerID() != this.containerID) {
            errStr = "Bad ContainerID field in Containerdata for " + this.containerID;
            errors.add(new ContainerScanError(ContainerScanError.FailureType.CORRUPT_CONTAINER_FILE, containerFile, new IOException(errStr)));
        }
        if (!(dbType = this.containerDataFromDisk.getContainerDBType()).equals("RocksDB")) {
            errStr = "Unknown DBType [" + dbType + "] in Container File for  [" + this.containerID + "]";
            errors.add(new ContainerScanError(ContainerScanError.FailureType.CORRUPT_CONTAINER_FILE, containerFile, new IOException(errStr)));
        }
        if (!this.metadataPath.equals(this.containerDataFromDisk.getMetadataPath())) {
            errStr = "Bad metadata path in Containerdata for " + this.containerID + "Expected [" + this.metadataPath + "] Got [" + this.containerDataFromDisk.getMetadataPath() + "]";
            errors.add(new ContainerScanError(ContainerScanError.FailureType.CORRUPT_CONTAINER_FILE, containerFile, new IOException(errStr)));
        }
        return errors;
    }

    private boolean containerIsDeleted() {
        return this.containerDataFromMemory.getState() == ContainerProtos.ContainerDataProto.State.DELETED;
    }

    private BlockData getBlockDataFromDB(DBHandle db, BlockData block) throws IOException {
        String blockKey = this.containerDataFromDisk.getBlockKey(block.getBlockID().getLocalID());
        return (BlockData)db.getStore().getBlockDataTable().get((Object)blockKey);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private boolean blockInDBWithLock(DBHandle db, BlockData block) throws IOException {
        this.container.readLock();
        try {
            boolean bl = this.getBlockDataFromDB(db, block) != null;
            return bl;
        }
        finally {
            this.container.readUnlock();
        }
    }

    private List<ContainerScanError> scanBlock(DBHandle db, File dbFile, BlockData block, DataTransferThrottler throttler, Canceler canceler, ContainerMerkleTreeWriter currentTree) {
        ContainerLayoutVersion layout = this.containerDataFromDisk.getLayoutVersion();
        ArrayList<ContainerScanError> blockErrors = new ArrayList<ContainerScanError>();
        boolean fileMissing = false;
        Iterator chunkIter = block.getChunks().iterator();
        while (chunkIter.hasNext() && !fileMissing) {
            ContainerProtos.ChunkInfo chunk = (ContainerProtos.ChunkInfo)chunkIter.next();
            Optional<Object> optionalFile = Optional.empty();
            try {
                optionalFile = Optional.of(layout.getChunkFile(this.containerDataFromDisk, block.getBlockID(), chunk.getChunkName()));
            }
            catch (StorageContainerException ex) {
                if (ex.getResult() == ContainerProtos.Result.UNABLE_TO_FIND_DATA_DIR) {
                    blockErrors.add(new ContainerScanError(ContainerScanError.FailureType.MISSING_CHUNKS_DIR, new File(this.containerDataFromDisk.getChunksPath()), (Exception)((Object)ex)));
                }
                blockErrors.add(new ContainerScanError(ContainerScanError.FailureType.CORRUPT_CHUNK, new File(this.containerDataFromDisk.getChunksPath()), (Exception)((Object)ex)));
            }
            if (optionalFile.isPresent()) {
                File chunkFile = (File)optionalFile.get();
                if (!chunkFile.exists()) {
                    if (!block.getChunks().isEmpty() && ((ContainerProtos.ChunkInfo)block.getChunks().get(0)).getLen() > 0L) {
                        ContainerScanError error = new ContainerScanError(ContainerScanError.FailureType.MISSING_DATA_FILE, new File(this.containerDataFromDisk.getChunksPath()), new IOException("Missing chunk file " + chunkFile.getAbsolutePath()));
                        blockErrors.add(error);
                    }
                } else if (chunk.getChecksumData().getType() != ContainerProtos.ChecksumType.NONE) {
                    currentTree.addBlock(block.getBlockID().getLocalID());
                    int bytesPerChecksum = chunk.getChecksumData().getBytesPerChecksum();
                    ByteBuffer buffer = BUFFER_POOL.getBuffer(bytesPerChecksum);
                    blockErrors.addAll(KeyValueContainerCheck.verifyChecksum(block, chunk, chunkFile, layout, buffer, currentTree, throttler, canceler));
                    buffer.clear();
                    BUFFER_POOL.returnBuffer(buffer);
                }
            }
            fileMissing = !optionalFile.isPresent() || !((File)optionalFile.get()).exists();
        }
        try {
            if (fileMissing && !this.blockInDBWithLock(db, block)) {
                blockErrors.clear();
                if (LOG.isDebugEnabled()) {
                    LOG.debug("Scanned outdated blockData {} in container {}", (Object)block, (Object)this.containerID);
                }
            }
        }
        catch (IOException ex) {
            blockErrors.add(new ContainerScanError(ContainerScanError.FailureType.INACCESSIBLE_DB, dbFile, ex));
        }
        return blockErrors;
    }

    private static List<ContainerScanError> verifyChecksum(BlockData block, ContainerProtos.ChunkInfo chunk, File chunkFile, ContainerLayoutVersion layout, ByteBuffer buffer, ContainerMerkleTreeWriter currentTree, DataTransferThrottler throttler, Canceler canceler) {
        ArrayList<ContainerScanError> scanErrors = new ArrayList<ContainerScanError>();
        ContainerProtos.ChunkInfo.Builder observedChunkBuilder = chunk.toBuilder();
        ContainerProtos.ChecksumData.Builder observedChecksumData = chunk.getChecksumData().toBuilder();
        observedChecksumData.clearChecksums();
        boolean chunkHealthy = true;
        boolean chunkMissing = false;
        ChecksumData checksumData = ChecksumData.getFromProtoBuf((ContainerProtos.ChecksumData)chunk.getChecksumData());
        int checksumCount = checksumData.getChecksums().size();
        int bytesPerChecksum = checksumData.getBytesPerChecksum();
        Checksum cal = new Checksum(checksumData.getChecksumType(), bytesPerChecksum);
        long bytesRead = 0L;
        try (FileChannel channel = FileChannel.open(chunkFile.toPath(), ChunkUtils.READ_OPTIONS, ChunkUtils.NO_ATTRIBUTES);){
            if (layout == ContainerLayoutVersion.FILE_PER_BLOCK) {
                channel.position(chunk.getOffset());
            }
            for (int i = 0; i < checksumCount; ++i) {
                int v;
                if (layout == ContainerLayoutVersion.FILE_PER_BLOCK && i == checksumCount - 1 && chunk.getLen() % (long)bytesPerChecksum != 0L) {
                    buffer.limit((int)(chunk.getLen() % (long)bytesPerChecksum));
                }
                if ((v = channel.read(buffer)) == -1) break;
                bytesRead += (long)v;
                buffer.flip();
                throttler.throttle((long)v, canceler);
                ByteString expected = (ByteString)checksumData.getChecksums().get(i);
                ByteString actual = (ByteString)cal.computeChecksum(buffer).getChecksums().get(0);
                observedChecksumData.addChecksums(actual);
                if (!chunkHealthy || expected.equals((Object)actual)) continue;
                String message = String.format("Inconsistent read for chunk=%s checksum item %d expected checksum %s actual checksum %s for block %s", ChunkInfo.getFromProtoBuf((ContainerProtos.ChunkInfo)chunk), i, StringUtils.bytes2Hex((ByteBuffer)expected.asReadOnlyByteBuffer()), StringUtils.bytes2Hex((ByteBuffer)actual.asReadOnlyByteBuffer()), block.getBlockID());
                chunkHealthy = false;
                scanErrors.add(new ContainerScanError(ContainerScanError.FailureType.CORRUPT_CHUNK, chunkFile, (Exception)new OzoneChecksumException(message)));
            }
            observedChunkBuilder.setLen(bytesRead);
            if (chunkHealthy && bytesRead != chunk.getLen()) {
                if (bytesRead == 0L) {
                    chunkMissing = true;
                    chunkHealthy = false;
                    String message = String.format("Missing chunk=%s with expected length=%d for block %s", chunk.getChunkName(), chunk.getLen(), block.getBlockID());
                    scanErrors.add(new ContainerScanError(ContainerScanError.FailureType.MISSING_CHUNK, chunkFile, new IOException(message)));
                } else {
                    String message = String.format("Inconsistent read for chunk=%s expected length=%d actual length=%d for block %s", chunk.getChunkName(), chunk.getLen(), bytesRead, block.getBlockID());
                    chunkHealthy = false;
                    scanErrors.add(new ContainerScanError(ContainerScanError.FailureType.INCONSISTENT_CHUNK_LENGTH, chunkFile, new IOException(message)));
                }
            }
        }
        catch (IOException ex) {
            chunkHealthy = false;
            scanErrors.add(new ContainerScanError(ContainerScanError.FailureType.CORRUPT_CHUNK, chunkFile, ex));
        }
        if (!chunkMissing) {
            observedChunkBuilder.setChecksumData(observedChecksumData);
            currentTree.addChunks(block.getBlockID().getLocalID(), chunkHealthy, observedChunkBuilder.build());
        }
        return scanErrors;
    }

    private void loadContainerData(File containerFile) throws IOException {
        this.containerDataFromDisk = (KeyValueContainerData)ContainerDataYaml.readContainerFile(containerFile);
        this.containerDataFromDisk.setVolume(this.volume);
        this.containerDataFromDisk.setDbFile(KeyValueContainerLocationUtil.getContainerDBFile(this.containerDataFromDisk));
    }
}

